/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

// If the display is jerky then reduce the value the for iteration limit.  This
// will reduce the calcultions needed for each pixel, but at the expense of the
// quality of generated image.

#define MAX_ITERATIONS 200 // Originaly 1024

/////////////////////////////////////////////////////////////////////////////////

//
// Description : Array and textureless GLSL 2D/3D/4D simplex 
//               noise functions.
//      Author : Ian McEwan, Ashima Arts.
//  Maintainer : ijm
//     Lastmod : 20110822 (ijm)
//     License : Copyright (C) 2011 Ashima Arts. All rights reserved.
//               Distributed under the MIT License. See LICENSE file.
//               https://github.com/ashima/webgl-noise
// 

vec3 mod289(vec3 x) {
  return x - floor(x * (1.0 / 289.0)) * 289.0;
}

vec4 mod289(vec4 x) {
  return x - floor(x * (1.0 / 289.0)) * 289.0;
}

vec4 permute(vec4 x) {
     return mod289(((x*34.0)+1.0)*x);
}

vec4 taylorInvSqrt(vec4 r)
{
  return 1.79284291400159 - 0.85373472095314 * r;
}

float snoise(vec3 v)
  { 
  const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;
  const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);

// First corner
  vec3 i  = floor(v + dot(v, C.yyy) );
  vec3 x0 =   v - i + dot(i, C.xxx) ;

// Other corners
  vec3 g = step(x0.yzx, x0.xyz);
  vec3 l = 1.0 - g;
  vec3 i1 = min( g.xyz, l.zxy );
  vec3 i2 = max( g.xyz, l.zxy );

  //   x0 = x0 - 0.0 + 0.0 * C.xxx;
  //   x1 = x0 - i1  + 1.0 * C.xxx;
  //   x2 = x0 - i2  + 2.0 * C.xxx;
  //   x3 = x0 - 1.0 + 3.0 * C.xxx;
  vec3 x1 = x0 - i1 + C.xxx;
  vec3 x2 = x0 - i2 + C.yyy; // 2.0*C.x = 1/3 = C.y
  vec3 x3 = x0 - D.yyy;      // -1.0+3.0*C.x = -0.5 = -D.y

// Permutations
  i = mod289(i); 
  vec4 p = permute( permute( permute( 
             i.z + vec4(0.0, i1.z, i2.z, 1.0 ))
           + i.y + vec4(0.0, i1.y, i2.y, 1.0 )) 
           + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));

// Gradients: 7x7 points over a square, mapped onto an octahedron.
// The ring size 17*17 = 289 is close to a multiple of 49 (49*6 = 294)
  float n_ = 0.142857142857; // 1.0/7.0
  vec3  ns = n_ * D.wyz - D.xzx;

  vec4 j = p - 49.0 * floor(p * ns.z * ns.z);  //  mod(p,7*7)

  vec4 x_ = floor(j * ns.z);
  vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)

  vec4 x = x_ *ns.x + ns.yyyy;
  vec4 y = y_ *ns.x + ns.yyyy;
  vec4 h = 1.0 - abs(x) - abs(y);

  vec4 b0 = vec4( x.xy, y.xy );
  vec4 b1 = vec4( x.zw, y.zw );

  //vec4 s0 = vec4(lessThan(b0,0.0))*2.0 - 1.0;
  //vec4 s1 = vec4(lessThan(b1,0.0))*2.0 - 1.0;
  vec4 s0 = floor(b0)*2.0 + 1.0;
  vec4 s1 = floor(b1)*2.0 + 1.0;
  vec4 sh = -step(h, vec4(0.0));

  vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;
  vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;

  vec3 p0 = vec3(a0.xy,h.x);
  vec3 p1 = vec3(a0.zw,h.y);
  vec3 p2 = vec3(a1.xy,h.z);
  vec3 p3 = vec3(a1.zw,h.w);

//Normalise gradients
  vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));
  p0 *= norm.x;
  p1 *= norm.y;
  p2 *= norm.z;
  p3 *= norm.w;

// Mix final noise value
  vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);
  m = m * m;
  return 42.0 * dot( m*m, vec4( dot(p0,x0), dot(p1,x1), 
                                dot(p2,x2), dot(p3,x3) ) );
  }

vec3 noiseSample(vec2 uv)
{	
	float f = 0.0;
    
    //uv *= 1.25;
    mat2 m = mat2( 1.6,  1.2, -1.2,  1.6 );
    f += 0.0625*snoise( vec3(uv ,iGlobalTime/2.0)); uv = m*uv;
    f += 1.3125*snoise( vec3(uv ,iGlobalTime/2.0)); uv = m*uv;
    f += 0.015725*snoise( vec3(uv ,iGlobalTime/2.0)); uv = m*uv;
    f += 0.00125*snoise( vec3(uv ,iGlobalTime/2.0)); uv = m*uv;
    f += 0.000525*snoise( vec3(uv ,iGlobalTime/2.0)); uv = m*uv;

	f = 0.5 + 0.5*f;
	
    return vec3(f,f,f);
}

// Created by inigo quilez - iq/2013
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

vec4 juliaSet(vec2 p)
{
    //p = p * 2.0 - 1.0;
    //p = p / ((p.x * p.x) + (p.y * p.y));

    /*vec2 cc = 1.1*vec2( 1.5*cos(0.1*iGlobalTime) - 0.25*cos(0.2*iGlobalTime), 
	                    1.5*sin(0.1*iGlobalTime) - 0.25*sin(0.2*iGlobalTime) );*/
    
    vec2 cc = 1.1*vec2( 0.5*cos(0.1*iGlobalTime) - 0.25*cos(0.2*iGlobalTime), 
	                    0.5*sin(0.1*iGlobalTime) - 0.25*sin(0.2*iGlobalTime) );
    
    /*vec2 cc = 1.1*vec2( 40.5*cos(0.1*iGlobalTime) - 20.25*cos(0.2*iGlobalTime), 
	                    40.5*sin(0.1*iGlobalTime) - 20.25*sin(0.2*iGlobalTime) );*/

	vec4 dmin = vec4(1000.0);
    vec2 z = (-1.0 + 2.0*p)*vec2(1.7,1.0);
    for( int i=0; i<MAX_ITERATIONS; i++ )
    {
      z = cc + vec2( z.x*z.x - z.y*z.y, 2.0*z.x*z.y );

		dmin=min(dmin, vec4(abs(0.0+z.y + 0.5*sin(z.x)), 
							abs(1.0+z.x + 0.5*sin(z.y)), 
							dot(z,z),
						    length( fract(z)-0.5) ) );
    }

	return dmin;
}

void main(void)
{
	vec2 p = gl_FragCoord.xy / iResolution.xy;
  
    vec4 dmin = juliaSet(p);
    
    float wave = sin(iGlobalTime) * 0.5 - 1.0;
    wave = 1000.0;
    
    vec3 color = vec3( dmin.w );
	color = mix( color, vec3(0.50,0.80,0.90),     min(1.0,pow(dmin.x*0.25,0.20)) );
    color = mix( color, vec3(0.072,0.070,0.060),     min(1.0,pow(dmin.y*0.50,0.50)) );
	color = mix( color, vec3(5.00,5.00,5.00), 1.0-min(1.0,pow(dmin.z*1.00,0.15) ));
  
	vec2 uv = p*vec2(iResolution.x/iResolution.y,1.0);
    vec3 noise = noiseSample(uv);
    
    color += noise / 6.0;
    
	color = 1.25*color*color;
	
	color *= 0.5 + 0.5*pow(16.0*p.x*(1.0-p.x)*p.y*(1.0-p.y),0.15); 
   
    gl_FragColor = vec4(color,1.0);
}